# Rock Paper Scissors - Unraid Deployment

This package contains all files needed to deploy the Rock Paper Scissors game with hand gesture recognition on Unraid.

## Features
- Real hand gesture recognition using MediaPipe
- Player registration and statistics tracking
- SQLite database for persistent storage
- Responsive web interface with animations
- Nginx reverse proxy setup
- HTTPS support with SSL certificates

## Directory Structure
```
unraid-deploy/
├── app/                     # Main application files
│   ├── index.html           # Main HTML file
│   ├── css/                 # Stylesheets
│   ├── js/                  # JavaScript files
│   ├── backend/             # Backend database files
│   ├── nginx/               # Nginx configuration
│   ├── package.json         # Node.js dependencies
│   ├── server.js            # Node.js server
│   └── Dockerfile           # Docker configuration
├── docker-compose.yml       # Docker Compose for Unraid
└── README.md                # This file
```

## Unraid Deployment Options

### Option 1: Individual Container Setup

#### Step 1: Create AppData Directory
1. Create directory on Unraid: `/mnt/user/appdata/rock-paper-scissors/`
2. Copy all files from `app/` folder to this directory

#### Step 2: Create Node.js Container
1. Go to Docker tab in Unraid
2. Click "Add Container"
3. Settings:
   - **Name**: `rock-paper-scissors-app`
   - **Repository**: `node:18-alpine`
   - **Network Type**: `Bridge`
   - **Console shell command**: `sh`
   - **Privileged**: `No`

4. **Port Mappings**:
   ```
   Host Port: 3001
   Container Port: 3001
   Connection Type: TCP
   ```

5. **Volume Mappings**:
   ```
   Host Path: /mnt/user/appdata/rock-paper-scissors
   Container Path: /app
   Access Mode: Read/Write
   ```

6. **Environment Variables**:
   ```
   NODE_ENV: production
   ```

7. **Post Script**:
   ```bash
   cd /app && npm install && npm start
   ```

#### Step 3: Create Nginx Container
1. Click "Add Container"
2. Settings:
   - **Name**: `rock-paper-scissors-nginx`
   - **Repository**: `nginx:alpine`
   - **Network Type**: `Bridge`
   - **Console shell command**: `/bin/sh`

3. **Port Mappings**:
   ```
   Host Port: 80
   Container Port: 80
   Connection Type: TCP
   
   Host Port: 443
   Container Port: 443
   Connection Type: TCP
   ```

4. **Volume Mappings**:
   ```
   Host Path: /mnt/user/appdata/rock-paper-scissors/nginx/rock-paper-scissors.conf
   Container Path: /etc/nginx/conf.d/default.conf
   Access Mode: Read Only
   
   Host Path: /mnt/user/appdata/rock-paper-scissors
   Container Path: /var/www/rock-paper-scissors
   Access Mode: Read Only
   
   Host Path: /mnt/user/appdata/rock-paper-scissors/ssl
   Container Path: /etc/nginx/ssl
   Access Mode: Read Only
   ```

5. **Extra Parameters**:
   ```
   --link rock-paper-scissors-app:app
   ```

### Option 2: Docker Compose (if Community Applications plugin is installed)

1. Install "Docker Compose" plugin from Community Applications
2. Copy `docker-compose.yml` to `/mnt/user/appdata/rock-paper-scissors/`
3. Run: `docker-compose up -d`

## SSL Certificate Setup

### Generate Self-Signed Certificates
1. SSH into your Unraid server
2. Navigate to app directory:
   ```bash
   cd /mnt/user/appdata/rock-paper-scissors
   ```
3. Run the SSL generation script:
   ```bash
   chmod +x generate-ssl.sh
   ./generate-ssl.sh
   ```

### Use Your Own SSL Certificates
1. Place your certificate files in `/mnt/user/appdata/rock-paper-scissors/ssl/`
2. Rename them to `cert.pem` and `key.pem`
3. Update permissions:
   ```bash
   chmod 644 ssl/cert.pem
   chmod 600 ssl/key.pem
   ```

## File Permissions

Set proper permissions for the application:
```bash
cd /mnt/user/appdata/rock-paper-scissors
chmod -R 755 .
chmod 775 backend/database
chown -R 1000:1000 .  # Optional: match container user
```

## Accessing the Application

After deployment, access the game at:
- HTTP: `http://YOUR_UNRAID_IP`
- HTTPS: `https://YOUR_UNRAID_IP` (after SSL setup)

## Game Instructions

1. Enter your name and click "Start Game"
2. Allow camera access when prompted
3. Show hand gestures to the camera:
   - Rock: Closed fist
   - Paper: Open hand (all fingers extended)
   - Scissors: Two fingers extended (V-shape)
4. Click "Play" to start a round
5. During the 3-second countdown, show your gesture
6. The computer will randomly select its gesture
7. Results are displayed with animations

## Troubleshooting

### Webcam Not Working
1. Ensure you're accessing via HTTPS (not HTTP)
2. Check browser permissions for camera access
3. Verify no other application is using the camera

### Database Issues
1. Check permissions on `backend/database/` directory
2. Ensure the directory is writable by the container

### Container Won't Start
1. Check Unraid logs for error messages
2. Verify all file paths in volume mappings exist
3. Ensure ports are not already in use

### Performance Issues
1. The hand detection requires processing power
2. Consider using a system with decent CPU for smooth experience

## Updating the Application

To update:
1. Stop the containers
2. Replace files in `/mnt/user/appdata/rock-paper-scissors/`
3. Start the containers
4. If package.json changed, run `npm install` in the container

## Customization

### Change Ports
Edit docker-compose.yml or container settings to use different ports

### Modify Game Rules
Edit `js/main.js` - modify `determineWinner()` function

### Change Styling
Edit `css/style.css` for visual modifications

## Support

For issues with this deployment package, please check:
1. All files are correctly copied
2. Permissions are properly set
3. SSL certificates are correctly generated
4. Container logs for specific error messages
