const sqlite3 = require('sqlite3').verbose();
const path = require('path');

// Create or connect to the database
const dbPath = path.join(__dirname, 'rps_game.db');
const db = new sqlite3.Database(dbPath);

// Initialize the database tables
db.serialize(() => {
    // Create players table
    db.run(`CREATE TABLE IF NOT EXISTS players (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        name TEXT UNIQUE NOT NULL,
        created_at DATETIME DEFAULT CURRENT_TIMESTAMP
    )`);
    
    // Create game_logs table
    db.run(`CREATE TABLE IF NOT EXISTS game_logs (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        player_id INTEGER,
        player_gesture TEXT NOT NULL,
        computer_gesture TEXT NOT NULL,
        result TEXT NOT NULL,
        played_at DATETIME DEFAULT CURRENT_TIMESTAMP,
        FOREIGN KEY (player_id) REFERENCES players (id)
    )`);
    
    // Create player_stats table to store win/loss/draw counts
    db.run(`CREATE TABLE IF NOT EXISTS player_stats (
        player_id INTEGER PRIMARY KEY,
        wins INTEGER DEFAULT 0,
        losses INTEGER DEFAULT 0,
        draws INTEGER DEFAULT 0,
        FOREIGN KEY (player_id) REFERENCES players (id)
    )`);
});

console.log('Database initialized successfully');

// Function to register a new player
function registerPlayer(name, callback) {
    const stmt = db.prepare('INSERT OR IGNORE INTO players (name) VALUES (?)');
    stmt.run(name, function(err) {
        if (err) {
            return callback(err);
        }
        
        // If player was inserted, also create stats record
        if (this.changes > 0) {
            const statsStmt = db.prepare('INSERT INTO player_stats (player_id) VALUES (?)');
            statsStmt.run(this.lastID, (err) => {
                callback(err, {id: this.lastID, name: name});
            });
        } else {
            // If player already exists, get their info
            db.get('SELECT id, name FROM players WHERE name = ?', [name], (err, row) => {
                callback(err, row);
            });
        }
    });
    stmt.finalize();
}

// Function to log a game result
function logGame(playerId, playerGesture, computerGesture, result, callback) {
    const stmt = db.prepare('INSERT INTO game_logs (player_id, player_gesture, computer_gesture, result) VALUES (?, ?, ?, ?)');
    stmt.run([playerId, playerGesture, computerGesture, result], function(err) {
        if (err) {
            return callback(err);
        }
        
        // Update player stats
        let statColumn;
        switch(result) {
            case 'win': statColumn = 'wins'; break;
            case 'lose': statColumn = 'losses'; break;
            case 'draw': statColumn = 'draws'; break;
        }
        
        if (statColumn) {
            const updateStmt = db.prepare(`UPDATE player_stats SET ${statColumn} = ${statColumn} + 1 WHERE player_id = ?`);
            updateStmt.run([playerId], (err) => {
                callback(err, {id: this.lastID});
            });
            updateStmt.finalize();
        } else {
            callback(null, {id: this.lastID});
        }
    });
    stmt.finalize();
}

// Function to get player statistics
function getPlayerStats(playerId, callback) {
    db.get('SELECT wins, losses, draws FROM player_stats WHERE player_id = ?', [playerId], callback);
}

// Function to get player game history
function getPlayerHistory(playerId, limit, callback) {
    const query = `
        SELECT player_gesture, computer_gesture, result, played_at 
        FROM game_logs 
        WHERE player_id = ? 
        ORDER BY played_at DESC 
        LIMIT ?
    `;
    db.all(query, [playerId, limit || 20], callback);
}

module.exports = {
    db,
    registerPlayer,
    logGame,
    getPlayerStats,
    getPlayerHistory
};
