// Game elements
const registrationScreen = document.getElementById('registration-screen');
const gameScreen = document.getElementById('game-screen');
const playerNameInput = document.getElementById('player-name');
const startGameButton = document.getElementById('start-game');
const playerNameDisplay = document.getElementById('player-name-display');
const playButton = document.getElementById('play-button');
const playAgainButton = document.getElementById('play-again');
const countdownElement = document.getElementById('countdown');
const playerGesture = document.getElementById('player-gesture');
const computerGesture = document.getElementById('computer-gesture');
const detectedGesture = document.getElementById('detected-gesture');
const computerGestureLabel = document.getElementById('computer-gesture-label');
const resultMessage = document.getElementById('result-message');
const winsElement = document.getElementById('wins');
const lossesElement = document.getElementById('losses');
const drawsElement = document.getElementById('draws');
const gameLog = document.getElementById('game-log');
const webcamElement = document.getElementById('webcam');
const gestureCanvas = document.getElementById('gesture-canvas');
const animationsContainer = document.getElementById('animations-container');
const logoutButton = document.getElementById('logout-button');

// Game state
let playerName = '';
let playerId = null;
let wins = 0;
let losses = 0;
let draws = 0;
let gameLogEntries = [];
let webcamStream = null;
let gestureContext = gestureCanvas.getContext('2d');

// MediaPipe Hands
let hands = null;
let camera = null;
let currentGesture = 'unknown';

// API endpoints
const API_BASE = '/api';

// Gesture mapping
const gestureEmojis = {
    'rock': '✊',
    'paper': '✋',
    'scissors': '✌️',
    'unknown': '?'
};

// Initialize the game
function initGame() {
    startGameButton.addEventListener('click', startGame);
    playButton.addEventListener('click', startCountdown);
    playAgainButton.addEventListener('click', resetGame);
    logoutButton.addEventListener('click', logout);
    
    // Set canvas size to match video
    gestureCanvas.width = playerGesture.offsetWidth;
    gestureCanvas.height = playerGesture.offsetHeight;
}

// Logout function
function logout() {
    // Stop webcam stream if it's active
    if (webcamStream) {
        webcamStream.getTracks().forEach(track => track.stop());
        webcamStream = null;
    }
    
    // Clear game state
    playerName = '';
    playerId = null;
    wins = 0;
    losses = 0;
    draws = 0;
    gameLogEntries = [];
    
    // Reset UI elements
    winsElement.textContent = '0';
    lossesElement.textContent = '0';
    drawsElement.textContent = '0';
    gameLog.innerHTML = '';
    playerNameDisplay.textContent = '';
    playerNameInput.value = '';
    playButton.disabled = false;
    playAgainButton.classList.add('hidden');
    countdownElement.classList.add('hidden');
    resultMessage.textContent = '';
    computerGesture.textContent = '?';
    computerGestureLabel.textContent = '-';
    detectedGesture.textContent = '-';
    detectedGesture.dataset.gesture = '';
    
    // Clear animations
    animationsContainer.innerHTML = '';
    
    // Switch back to registration screen
    gameScreen.classList.add('hidden');
    registrationScreen.classList.remove('hidden');
}

// Start the game after player registration
async function startGame() {
    playerName = playerNameInput.value.trim();
    if (!playerName) {
        alert('Please enter your name');
        return;
    }
    
    try {
        // Register player with backend
        const response = await fetch(`${API_BASE}/register`, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify({ name: playerName })
        });
        
        const data = await response.json();
        if (!response.ok) {
            throw new Error(data.error || 'Failed to register player');
        }
        
        playerId = data.player.id;
        playerNameDisplay.textContent = playerName;
        registrationScreen.classList.add('hidden');
        gameScreen.classList.remove('hidden');
        
        // Load player stats
        await loadPlayerStats();
        
        // Load player history
        await loadPlayerHistory();
        
        // Initialize webcam
        initWebcam();
    } catch (error) {
        console.error('Error registering player:', error);
        alert('Failed to register player. Please try again.');
    }
}

// Initialize webcam for gesture detection
function initWebcam() {
    if (navigator.mediaDevices && navigator.mediaDevices.getUserMedia) {
        navigator.mediaDevices.getUserMedia({ video: true })
            .then(function(stream) {
                webcamStream = stream;
                webcamElement.srcObject = stream;
                detectedGesture.textContent = " Webcam active - show gesture";
                // Initialize MediaPipe Hands
                initializeHandDetection();
            })
            .catch(function(error) {
                console.error("Error accessing webcam:", error);
                detectedGesture.textContent = "Webcam error: " + error.message;
                if (error.name === 'NotAllowedError') {
                    detectedGesture.textContent += " (Permission denied)";
                } else if (error.name === 'NotFoundError') {
                    detectedGesture.textContent += " (No camera found)";
                } else if (error.name === 'NotReadableError') {
                    detectedGesture.textContent += " (Camera in use)";
                } else if (error.name === 'OverconstrainedError') {
                    detectedGesture.textContent += " (Camera not suitable)";
                }
            });
    } else {
        detectedGesture.textContent = "Webcam not supported by this browser";
        console.error("Webcam not supported");
    }
}

// Initialize MediaPipe Hands for gesture detection
function initializeHandDetection() {
    // Create MediaPipe Hands instance
    hands = new Hands({
        locateFile: (file) => {
            return `https://cdn.jsdelivr.net/npm/@mediapipe/hands/${file}`;
        }
    });
    
    hands.setOptions({
        maxNumHands: 1,
        modelComplexity: 1,
        minDetectionConfidence: 0.5,
        minTrackingConfidence: 0.5
    });
    
    hands.onResults(onHandResults);
    
    // Create camera instance
    camera = new Camera(webcamElement, {
        onFrame: async () => {
            await hands.send({image: webcamElement});
        },
        width: 640,
        height: 480
    });
    
    camera.start();
}

// Process hand detection results
function onHandResults(results) {
    // Clear canvas
    gestureContext.clearRect(0, 0, gestureCanvas.width, gestureCanvas.height);
    
    if (results.multiHandLandmarks && results.multiHandLandmarks.length > 0) {
        // Draw hand landmarks
        for (const landmarks of results.multiHandLandmarks) {
            drawConnectors(gestureContext, landmarks, HAND_CONNECTIONS, {color: '#00FF00', lineWidth: 2});
            drawLandmarks(gestureContext, landmarks, {color: '#FF0000', lineWidth: 1});
        }
        
        // Detect gesture
        const gesture = detectGesture(results.multiHandLandmarks[0]);
        currentGesture = gesture;
        
        // Only update display when not in countdown
        if (!countdownElement.classList.contains('hidden')) return;
        
        detectedGesture.textContent = gestureEmojis[gesture];
        detectedGesture.dataset.gesture = gesture;
    } else {
        // No hand detected
        currentGesture = 'unknown';
        if (!countdownElement.classList.contains('hidden')) return;
        detectedGesture.textContent = gestureEmojis['unknown'];
        detectedGesture.dataset.gesture = 'unknown';
    }
}

// Detect gesture based on hand landmarks
function detectGesture(landmarks) {
    if (!landmarks) return 'unknown';
    
    // Get key finger landmarks
    const thumbTip = landmarks[4];
    const indexTip = landmarks[8];
    const middleTip = landmarks[12];
    const ringTip = landmarks[16];
    const pinkyTip = landmarks[20];
    
    const thumbMcp = landmarks[2];
    const indexMcp = landmarks[5];
    const middleMcp = landmarks[9];
    const ringMcp = landmarks[13];
    const pinkyMcp = landmarks[17];
    
    // Calculate distances (simplified gesture detection)
    const indexExtended = indexTip.y < indexMcp.y;
    const middleExtended = middleTip.y < middleMcp.y;
    const ringExtended = ringTip.y < ringMcp.y;
    const pinkyExtended = pinkyTip.y < pinkyMcp.y;
    const thumbExtended = thumbTip.x < thumbMcp.x || thumbTip.x > thumbMcp.x; // Simplified
    
    // Rock (closed fist) - all fingers curled
    if (!indexExtended && !middleExtended && !ringExtended && !pinkyExtended) {
        return 'rock';
    }
    
    // Paper (open hand) - all fingers extended
    if (indexExtended && middleExtended && ringExtended && pinkyExtended) {
        return 'paper';
    }
    
    // Scissors (two fingers) - index and middle extended, others curled
    if (indexExtended && middleExtended && !ringExtended && !pinkyExtended) {
        return 'scissors';
    }
    
    return 'unknown';
}

// Start the countdown when play button is clicked
function startCountdown() {
    playButton.disabled = true;
    countdownElement.classList.remove('hidden');
    resultMessage.textContent = '';
    
    let count = 3;
    countdownElement.textContent = count;
    
    const countdownInterval = setInterval(() => {
        count--;
        if (count > 0) {
            countdownElement.textContent = count;
        } else {
            clearInterval(countdownInterval);
            countdownElement.classList.add('hidden');
            playGame();
        }
    }, 1000);
}

// Main game logic
async function playGame() {
    // Get player's gesture (from MediaPipe detection)
    const playerGestureName = currentGesture || 'unknown';
    
    // Computer randomly chooses gesture
    const gestures = ['rock', 'paper', 'scissors'];
    const computerGestureName = gestures[Math.floor(Math.random() * gestures.length)];
    
    // Display computer's gesture
    computerGesture.textContent = gestureEmojis[computerGestureName];
    computerGestureLabel.textContent = computerGestureName;
    
    // Determine winner
    const result = determineWinner(playerGestureName, computerGestureName);
    
    // Display result
    displayResult(result, playerGestureName, computerGestureName);
    
    // Update statistics
    updateStatistics(result);
    
    // Add to game log
    await addToGameLog(playerGestureName, computerGestureName, result);
    
    // Show play again button
    playAgainButton.classList.remove('hidden');
    
    // Play animations
    playAnimations(result);
}

// Determine the winner
function determineWinner(player, computer) {
    if (player === computer) {
        return 'draw';
    }
    
    if (
        (player === 'rock' && computer === 'scissors') ||
        (player === 'paper' && computer === 'rock') ||
        (player === 'scissors' && computer === 'paper')
    ) {
        return 'win';
    }
    
    return 'lose';
}

// Display the result
function displayResult(result, playerGesture, computerGesture) {
    switch (result) {
        case 'win':
            resultMessage.textContent = 'You Win!';
            resultMessage.style.color = '#4caf50';
            break;
        case 'lose':
            resultMessage.textContent = 'You Lose!';
            resultMessage.style.color = '#f44336';
            break;
        case 'draw':
            resultMessage.textContent = 'It\'s a Draw!';
            resultMessage.style.color = '#ff9800';
            break;
        default:
            resultMessage.textContent = 'Game Error!';
            resultMessage.style.color = '#9e9e9e';
    }
}

// Update game statistics
function updateStatistics(result) {
    switch (result) {
        case 'win':
            wins++;
            winsElement.textContent = wins;
            break;
        case 'lose':
            losses++;
            lossesElement.textContent = losses;
            break;
        case 'draw':
            draws++;
            drawsElement.textContent = draws;
            break;
    }
}

// Add entry to game log
async function addToGameLog(playerGesture, computerGesture, result) {
    const date = new Date();
    const timeString = date.toLocaleTimeString();
    
    const resultText = result === 'win' ? 'Won' : result === 'lose' ? 'Lost' : 'Draw';
    
    // Update the log display
    const logEntryElement = document.createElement('div');
    logEntryElement.className = 'log-entry';
    
    const playerEmoji = gestureEmojis[playerGesture] || '?';
    const computerEmoji = gestureEmojis[computerGesture] || '?';
    
    logEntryElement.innerHTML = `
        ${timeString} | ${playerEmoji} vs ${computerEmoji} | ${resultText}
    `;
    
    gameLog.insertBefore(logEntryElement, gameLog.firstChild);
    
    // Log to backend
    await logGameToBackend(playerGesture, computerGesture, result);
}

// Play animations based on result
function playAnimations(result) {
    switch (result) {
        case 'win':
            playConfetti();
            break;
        case 'lose':
            playSadFaces();
            break;
        case 'draw':
            playSwordClash();
            break;
    }
}

// Play confetti animation
function playConfetti() {
    for (let i = 0; i < 150; i++) {
        setTimeout(() => {
            const confetti = document.createElement('div');
            confetti.className = 'animation-element';
            confetti.textContent = '🎉';
            confetti.style.left = Math.random() * 100 + 'vw';
            confetti.style.top = '-50px';
            confetti.style.fontSize = (Math.random() * 20 + 15) + 'px';
            animationsContainer.appendChild(confetti);
            
            // Animate falling
            const animation = confetti.animate([
                { transform: 'translateY(0) rotate(0deg)', opacity: 1 },
                { transform: `translateY(${window.innerHeight}px) rotate(${Math.random() * 360}deg)`, opacity: 0.7 }
            ], {
                duration: 3000,
                easing: 'cubic-bezier(0.1, 0.8, 0.2, 1)'
            });
            
            animation.onfinish = () => {
                confetti.remove();
            };
        }, i * 20);
    }
}

// Play sad faces animation
function playSadFaces() {
    for (let i = 0; i < 50; i++) {
        setTimeout(() => {
            const sadFace = document.createElement('div');
            sadFace.className = 'animation-element';
            sadFace.textContent = '😢';
            sadFace.style.left = Math.random() * 100 + 'vw';
            sadFace.style.top = '-50px';
            sadFace.style.fontSize = (Math.random() * 20 + 15) + 'px';
            animationsContainer.appendChild(sadFace);
            
            // Animate falling
            const animation = sadFace.animate([
                { transform: 'translateY(0)', opacity: 1 },
                { transform: `translateY(${window.innerHeight}px)`, opacity: 0.7 }
            ], {
                duration: 2000,
                easing: 'cubic-bezier(0.1, 0.8, 0.2, 1)'
            });
            
            animation.onfinish = () => {
                sadFace.remove();
            };
        }, i * 40);
    }
}

// Play sword clash animation
function playSwordClash() {
    const clash = document.createElement('div');
    clash.className = 'animation-element';
    clash.textContent = '⚔️💥';
    clash.style.left = '50%';
    clash.style.top = '50%';
    clash.style.fontSize = '5rem';
    clash.style.transform = 'translate(-50%, -50%)';
    clash.style.zIndex = '1000';
    animationsContainer.appendChild(clash);
    
    // Add animation effect
    const animation = clash.animate([
        { transform: 'translate(-50%, -50%) scale(0.5)', opacity: 0 },
        { transform: 'translate(-50%, -50%) scale(1.5)', opacity: 1 },
        { transform: 'translate(-50%, -50%) scale(1)', opacity: 1 },
        { transform: 'translate(-50%, -50%) scale(1.2)', opacity: 0.7 },
        { transform: 'translate(-50%, -50%) scale(0.8)', opacity: 0 }
    ], {
        duration: 2000,
        easing: 'ease-in-out'
    });
    
    animation.onfinish = () => {
        clash.remove();
    };
}

// Reset the game for another round
function resetGame() {
    playButton.disabled = false;
    playAgainButton.classList.add('hidden');
    resultMessage.textContent = '';
    computerGesture.textContent = '?';
    computerGestureLabel.textContent = '-';
    
    // Clear any animations
    animationsContainer.innerHTML = '';
}

// Clean up when the page is closed
window.addEventListener('beforeunload', () => {
    if (webcamStream) {
        webcamStream.getTracks().forEach(track => track.stop());
    }
    
    // Stop MediaPipe camera if active
    if (camera) {
        camera.stop();
    }
});

// Load player statistics from backend
async function loadPlayerStats() {
    if (!playerId) return;
    
    try {
        const response = await fetch(`${API_BASE}/player/${playerId}/stats`);
        const data = await response.json();
        
        if (response.ok && data.stats) {
            wins = data.stats.wins || 0;
            losses = data.stats.losses || 0;
            draws = data.stats.draws || 0;
            
            winsElement.textContent = wins;
            lossesElement.textContent = losses;
            drawsElement.textContent = draws;
        }
    } catch (error) {
        console.error('Error loading player stats:', error);
    }
}

// Load player game history from backend
async function loadPlayerHistory() {
    if (!playerId) return;
    
    try {
        const response = await fetch(`${API_BASE}/player/${playerId}/history?limit=20`);
        const data = await response.json();
        
        if (response.ok && data.history) {
            gameLog.innerHTML = '';
            data.history.forEach(entry => {
                const logEntryElement = document.createElement('div');
                logEntryElement.className = 'log-entry';
                
                const playerEmoji = gestureEmojis[entry.player_gesture] || '?';
                const computerEmoji = gestureEmojis[entry.computer_gesture] || '?';
                
                // Format the date
                const date = new Date(entry.played_at);
                const timeString = date.toLocaleTimeString();
                
                logEntryElement.innerHTML = `
                    ${timeString} | ${playerEmoji} vs ${computerEmoji} | ${entry.result}
                `;
                
                gameLog.insertBefore(logEntryElement, gameLog.firstChild);
            });
        }
    } catch (error) {
        console.error('Error loading player history:', error);
    }
}

// Log game result to backend
async function logGameToBackend(playerGesture, computerGesture, result) {
    if (!playerId) return;
    
    try {
        const response = await fetch(`${API_BASE}/log-game`, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify({
                playerId: playerId,
                playerGesture: playerGesture,
                computerGesture: computerGesture,
                result: result
            })
        });
        
        const data = await response.json();
        if (!response.ok) {
            throw new Error(data.error || 'Failed to log game');
        }
        
        // Reload stats after logging
        await loadPlayerStats();
        await loadPlayerHistory();
    } catch (error) {
        console.error('Error logging game:', error);
    }
}

// Initialize the game when the page loads
document.addEventListener('DOMContentLoaded', initGame);
