const express = require('express');
const path = require('path');
const cors = require('cors');
const { registerPlayer, logGame, getPlayerStats, getPlayerHistory } = require('./backend/database/db');

const app = express();
const PORT = process.env.PORT || 3001;

// Middleware
app.use(cors());
app.use(express.json());
app.use(express.static(path.join(__dirname, '.')));

// API Routes

// Register a new player
app.post('/api/register', (req, res) => {
    const { name } = req.body;
    
    if (!name) {
        return res.status(400).json({ error: 'Player name is required' });
    }
    
    registerPlayer(name, (err, player) => {
        if (err) {
            return res.status(500).json({ error: 'Failed to register player' });
        }
        res.json({ player });
    });
});

// Log a game result
app.post('/api/log-game', (req, res) => {
    const { playerId, playerGesture, computerGesture, result } = req.body;
    
    if (!playerId || !playerGesture || !computerGesture || !result) {
        return res.status(400).json({ error: 'Missing required fields' });
    }
    
    logGame(playerId, playerGesture, computerGesture, result, (err, log) => {
        if (err) {
            return res.status(500).json({ error: 'Failed to log game' });
        }
        res.json({ log });
    });
});

// Get player statistics
app.get('/api/player/:id/stats', (req, res) => {
    const playerId = req.params.id;
    
    getPlayerStats(playerId, (err, stats) => {
        if (err) {
            return res.status(500).json({ error: 'Failed to retrieve stats' });
        }
        res.json({ stats });
    });
});

// Get player game history
app.get('/api/player/:id/history', (req, res) => {
    const playerId = req.params.id;
    const limit = parseInt(req.query.limit) || 20;
    
    getPlayerHistory(playerId, limit, (err, history) => {
        if (err) {
            return res.status(500).json({ error: 'Failed to retrieve history' });
        }
        res.json({ history });
    });
});

// Serve the main game page
app.get('/', (req, res) => {
    res.sendFile(path.join(__dirname, 'index.html'));
});

// Start the server
app.listen(PORT, () => {
    console.log(`Server is running on http://localhost:${PORT}`);
});
